import React, { useState } from 'react';
import { X, Plus, Trash2 } from 'lucide-react';

export function Settings({ settings, onSave, onClose }) {
    const [localSettings, setLocalSettings] = useState({
        provider: 'gemini', // default
        longcatApiKey: '',
        ...settings
    });
    const [newKey, setNewKey] = useState('');

    const handleAddKey = () => {
        if (!newKey.trim()) return;
        setLocalSettings({
            ...localSettings,
            apiKeys: [...(localSettings.apiKeys || []), newKey.trim()]
        });
        setNewKey('');
    };

    const handleRemoveKey = (index) => {
        setLocalSettings({
            ...localSettings,
            apiKeys: localSettings.apiKeys.filter((_, i) => i !== index)
        });
    };

    const handleSave = () => {
        onSave(localSettings);
    };

    const isLongcat = localSettings.provider === 'longcat';

    // Helper to change provider and reset model to default for that provider
    const changeProvider = (newProvider) => {
        let newModel = localSettings.model;
        if (newProvider === 'longcat') {
            newModel = 'LongCat-Flash-Chat';
        } else {
            newModel = 'gemini-flash-latest';
        }
        setLocalSettings({ ...localSettings, provider: newProvider, model: newModel });
    };

    return (
        <div className="modal-overlay">
            <div className="modal-content">
                <div className="modal-header">
                    <h2>Settings</h2>
                    <button onClick={onClose} className="btn-icon">
                        <X size={20} />
                    </button>
                </div>

                <div className="modal-body">
                    {/* Provider Selection */}
                    <div className="settings-section">
                        <h3>AI Provider</h3>
                        <div className="flex gap-4">
                            <label className="flex items-center gap-2 cursor-pointer">
                                <input
                                    type="radio"
                                    name="provider"
                                    value="gemini"
                                    checked={localSettings.provider !== 'longcat'}
                                    onChange={() => changeProvider('gemini')}
                                />
                                Google Gemini
                            </label>
                            <label className="flex items-center gap-2 cursor-pointer">
                                <input
                                    type="radio"
                                    name="provider"
                                    value="longcat"
                                    checked={localSettings.provider === 'longcat'}
                                    onChange={() => changeProvider('longcat')}
                                />
                                Longcat AI
                            </label>
                        </div>
                    </div>

                    {isLongcat ? (
                        /* Longcat Settings */
                        <>
                            <div className="settings-section">
                                <h3>Longcat API Key</h3>
                                <input
                                    type="password"
                                    value={localSettings.longcatApiKey || ''}
                                    onChange={(e) => setLocalSettings({ ...localSettings, longcatApiKey: e.target.value })}
                                    placeholder="Enter Longcat API Key"
                                    className="input w-full"
                                />
                            </div>
                            <div className="settings-section">
                                <h3>Model (Longcat)</h3>
                                <select
                                    value={localSettings.model}
                                    onChange={(e) => setLocalSettings({ ...localSettings, model: e.target.value })}
                                    className="select"
                                >
                                    <option value="LongCat-Flash-Chat">LongCat-Flash-Chat</option>
                                    <option value="LongCat-Flash-Thinking">LongCat-Flash-Thinking</option>
                                    <option value="LongCat-Flash-Thinking-2601">LongCat-Flash-Thinking-2601</option>
                                    <option value="LongCat-Flash-Lite">LongCat-Flash-Lite</option>
                                </select>
                            </div>
                        </>
                    ) : (
                        /* Gemini Settings */
                        <>
                            <div className="settings-section">
                                <h3>API Keys (Gemini)</h3>
                                <p className="text-muted">Add multiple keys for rotation.</p>

                                <div className="api-keys-list">
                                    {(localSettings.apiKeys || []).map((key, i) => (
                                        <div key={i} className="api-key-item">
                                            <span className="api-key-text">{key.substring(0, 8)}...{key.substring(key.length - 4)}</span>
                                            <button onClick={() => handleRemoveKey(i)} className="btn-icon-small">
                                                <Trash2 size={14} />
                                            </button>
                                        </div>
                                    ))}
                                </div>

                                <div className="api-key-input-group">
                                    <input
                                        type="text"
                                        value={newKey}
                                        onChange={(e) => setNewKey(e.target.value)}
                                        placeholder="Enter Gemini API Key"
                                        className="input"
                                        onKeyDown={(e) => e.key === 'Enter' && handleAddKey()}
                                    />
                                    <button onClick={handleAddKey} className="btn-secondary">
                                        <Plus size={16} /> Add
                                    </button>
                                </div>
                            </div>

                            <div className="settings-section">
                                <h3>Model (Gemini)</h3>
                                <select
                                    value={localSettings.model}
                                    onChange={(e) => setLocalSettings({ ...localSettings, model: e.target.value })}
                                    className="select"
                                >
                                    <option value="gemini-flash-latest">gemini-flash-latest</option>
                                    <option value="gemini-2.0-flash-exp">gemini-2.0-flash-exp</option>
                                    <option value="gemini-1.5-flash">gemini-1.5-flash</option>
                                    <option value="gemini-1.5-pro">gemini-1.5-pro</option>
                                </select>
                            </div>
                        </>
                    )}

                    {/* Temperature */}
                    <div className="settings-section">
                        <h3>Temperature: {localSettings.temperature}</h3>
                        <input
                            type="range"
                            min="0"
                            max="2"
                            step="0.1"
                            value={localSettings.temperature}
                            onChange={(e) => setLocalSettings({ ...localSettings, temperature: parseFloat(e.target.value) })}
                            className="slider"
                        />
                    </div>
                </div>

                <div className="modal-footer">
                    <button onClick={onClose} className="btn-secondary">Cancel</button>
                    <button onClick={handleSave} className="btn-primary">Save</button>
                </div>
            </div>
        </div>
    );
}
